const gateway = Gateway.create(public_key);
const threeDS = gateway.get3DSecure();

// ===================== Helpers: email sanitization/validation + 3DS deviceChannel =====================
function wolkSanitizeEmail(email) {
    return (email || '').toString().trim().toLowerCase();
}
function wolkIsValidEmail(email) {
    const e = wolkSanitizeEmail(email);
    // required: must be present and valid
    if (e === '') return false;
    return /^[^\s@]+@[^\s@]+\.[^\s@]+$/.test(e);
}

// Safe unmount para evitar error "Frame has not yet been mounted"
function safeUnmount(iface) {
    try {
        if (iface && typeof iface.unmount === 'function') {
            iface.unmount();
        }
    } catch (e) {
        console.log('Unmount skipped:', e.message);
    }
}
// ================================================================================================


// ===================== UI helpers (no "Approved" until WooCommerce confirms) =====================
function wolkSetTitleAndMessage(titleKeyOrText, messageKeyOrText) {
    const t = document.getElementById('responseTitle');
    const m = document.getElementById('response');
    const icon = document.getElementById('responseIcon');

    // Reset visual state
    if (icon) {
        icon.style.display = 'none';
        icon.classList.remove('is-error');
        icon.innerHTML = '';
    }

    if (t) t.innerHTML = (typeof titleKeyOrText === 'string') ? _lang(titleKeyOrText) : titleKeyOrText;
    if (m) m.innerHTML = (typeof messageKeyOrText === 'string') ? _lang(messageKeyOrText) : messageKeyOrText;
}

// Mostrar error con estilo minimalista: SOLO mensaje + X roja
function wolkShowInlineError(messageKeyOrText) {
    const t = document.getElementById('responseTitle');
    const m = document.getElementById('response');
    const icon = document.getElementById('responseIcon');

    if (t) {
        t.innerHTML = '';
        t.style.display = 'none';
    }

    if (icon) {
        icon.innerHTML = 'X';
        icon.classList.add('is-error');
        icon.style.display = 'block';
    }

    // Evitar mostrar "undefined" en UI
    if (messageKeyOrText === undefined || messageKeyOrText === null || String(messageKeyOrText).trim() === '') {
        messageKeyOrText = 'generalError';
    }

    if (m) {
        m.innerHTML = (typeof messageKeyOrText === 'string') ? _lang(messageKeyOrText) : messageKeyOrText;
    }
}

// ===================== ALERT ELEGANTE (reemplaza alert() nativo) =====================
function wolkShowAlert(message, type = 'warning') {
    const authBox = document.getElementsByTagName('authBox')[0];
    const loader = document.getElementById('loader');
    const t = document.getElementById('responseTitle');
    const m = document.getElementById('response');
    const icon = document.getElementById('responseIcon');
    const closeBtn = document.getElementById('close-modal');
    
    // Ocultar loader
    if (loader) loader.style.display = 'none';
    
    // Configurar icono segun tipo
    if (icon) {
        icon.style.display = 'flex';
        icon.classList.remove('is-error', 'is-success', 'is-warning');
        if (type === 'error') {
            icon.classList.add('is-error');
            icon.innerHTML = '✕';
        } else if (type === 'success') {
            icon.classList.add('is-success');
            icon.innerHTML = '✓';
        } else {
            // warning - usar amarillo/naranja
            icon.classList.add('is-warning');
            icon.innerHTML = '!';
        }
    }
    
    // Titulo segun tipo
    if (t) {
        t.style.display = 'block';
        if (type === 'error') {
            t.innerHTML = _lang('Error');
        } else if (type === 'success') {
            t.innerHTML = _lang('Success');
        } else {
            t.innerHTML = _lang('Attention');
        }
    }
    
    // Mensaje
    if (m) {
        m.innerHTML = (typeof message === 'string') ? _lang(message) : message;
    }
    
    // Mostrar boton cerrar
    if (closeBtn) closeBtn.style.display = 'flex';
    
    // Mostrar modal
    if (authBox) authBox.style.display = 'flex';
}

// Resolver mensaje de error UI de forma segura (evita undefined y clasifica casos comunes)
function wolkResolveUiError(res) {
    try {
        // 0) Si el backend ya clasifico el motivo, usalo (fuente de verdad para UI)
        const backendKey = (res && (res.wolk_reason_key || res.wolk_ui_key)) ? String(res.wolk_reason_key || res.wolk_ui_key).trim() : '';
        if (backendKey) return backendKey;

        // 1) Si el backend trae un codigo especifico (NO el status generico 1/2/3), usalo.
        //    Nota: en varios flujos `response_code` puede ser solo "2" o "3" (declined/error)
        //    y eso NO indica el motivo real. En esos casos preferimos clasificar por texto.
        const rc = (res && res.response_code !== undefined && res.response_code !== null) ? String(res.response_code).trim() : '';
        if (rc && rc !== '1' && rc !== '2' && rc !== '3') {
            return rc;
        }

        // 2) Si viene texto, clasificar (ej: tarjeta bloqueada/frozen)
        const rt = (res && (res.responsetext || res.response_text || res.message || res.wolk_reason)) ? String(res.responsetext || res.response_text || res.message || res.wolk_reason) : '';
        const t = rt.toLowerCase();

        if (
            t.includes('frozen') || t.includes('blocked') || t.includes('restricted') || t.includes('suspend') ||
            t.includes('bloquead') || t.includes('suspendid') || t.includes('restringid')
        ) {
            return 'cardFrozen';
        }
        if (
            t.includes('insufficient') || t.includes('fund') ||
            t.includes('fondos') || t.includes('saldo') || t.includes('insuficiente')
        ) {
            return 'insufficientFunds';
        }
        if (t.includes('expired')) {
            return 'cardExpired';
        }
        if (
            t.includes('cvv') || t.includes('cvc') || t.includes('cvv2') || t.includes('cvc2') ||
            t.includes('security code') || t.includes('verification') || t.includes('mismatch') ||
            t.includes('codigo de seguridad') || t.includes('codigo de seguridad')
        ) {
            return 'cvvInvalid';
        }

        // 3) Fallback
        return 'generalError';
    } catch (_) {
        return 'generalError';
    }
}

// Motivo minimalista para notas (si el backend todavia no lo anadio)
function wolkResolveMinimalReason(res) {
    const key = wolkResolveUiError(res);
    // Usar el texto ya traducido para que quede consistente en notas
    return _lang(key);
}

// Rehabilitar el titulo (cuando volvemos a estado normal/processing)
function wolkEnableTitle() {
    const t = document.getElementById('responseTitle');
    if (t) t.style.display = '';
}

// Registrar motivo en notas internas del pedido (trazabilidad)
function wolkAddOrderNote(minimalReason) {
    try {
        if (typeof order_id === 'undefined' || !order_id) return;
        if (!ajax_var || !ajax_var.url) return;

        const fd = new FormData();
        fd.append('action', 'wolk_add_order_note');
        fd.append('order_id', order_id);
        fd.append('message', minimalReason || 'Error de pago');
        fd.append('nonce', ajax_var.nonce);

        fetch(ajax_var.url, { method: 'POST', body: fd })
            .then(() => {})
            .catch(() => {});
    } catch (_) {
        // silencio
    }
}

function wolkShowProcessing() {
    // [OK] Requerimiento: NO decir "aprobada" hasta que el backend confirme el flujo
    wolkEnableTitle();
    wolkSetTitleAndMessage('Transaction In Process', 'Submitting Payment Request');
}

// Cuando el emisor requiere verificacion (3DS challenge), mantenemos el estado de "en proceso"
// y mostramos una instruccion clara (sin pantallas de "exito" intermedias).
function wolkShow3DSChallengeMessage() {
    wolkEnableTitle();
    wolkSetTitleAndMessage('Transaction In Process', '3dsVerificationRequired');
}

// Al cargar la pagina, mostramos el estado correcto (evita el falso "Approved" por cualquier motivo)
document.addEventListener('DOMContentLoaded', wolkShowProcessing);
// ================================================================================================

const queryString = window.location.search;
const urlParams = new URLSearchParams(queryString);
let threeDSecureInterface = null;

// ================================================================================================
// 3DS2 (NMI Gateway.js) - Manual Device Data Collection
// NMI docs specify these fields when you provide deviceChannel manually:
// browserJavaEnabled, browserJavascriptEnabled, browserLanguage, browserColorDepth,
// browserScreenHeight, browserScreenWidth, browserTimeZone, deviceChannel
// deviceChannel expected values include 'Browser'.
function wolkBuild3DSDeviceData() {
    let javaEnabled = 'false';
    try {
        // window.navigator.javaEnabled() is deprecated in modern browsers; wrap in try/catch.
        javaEnabled = String(window.navigator.javaEnabled());
    } catch (e) {
        javaEnabled = 'false';
    }
    return {
        browserJavaEnabled: javaEnabled,
        browserJavascriptEnabled: String(true),
        browserLanguage: (window.navigator.language || window.navigator.userLanguage || ''),
        browserColorDepth: String(window.screen.colorDepth || ''),
        browserScreenHeight: String(window.screen.height || ''),
        browserScreenWidth: String(window.screen.width || ''),
        browserTimeZone: String(new Date().getTimezoneOffset()),
        deviceChannel: 'Browser'
    };
}

gateway.on('error',(e)=>{
    // Verificar si el error es especificamente por 3DSecure inactivo
    if (e && e.message && e.message.includes("3DSecure is inactive on your account")) {
        // Mostrar mensaje especifico para el usuario/admin y el error tecnico
        show_3ds_inactive_error(e);
        create_error_log({
            details: e,
            type: '3DSecure-Inactive'
        });
    } else {
        // Para errores 3DS/gateway durante el flujo, mostrar SOLO el error (sin mensaje de "en proceso").
        try { document.getElementById('loader').style.display = 'none'; } catch(_) {}
        try { document.getElementsByTagName('authBox')[0].style.display = 'flex'; } catch(_) {}

        // Clasificacion minima para UI
        let uiKey = 'generalError';
        const msg = (e && e.message) ? String(e.message) : '';
        if (msg.includes('deviceChannel') || msg.includes('deviceRenderOptions') || msg.includes('sdkAppID')) {
            uiKey = '3dsRequiredDataMissing';
        } else if (e && e.type === 'gatewayError') {
            uiKey = 'gatewayError';
        }

        if (uiKey === '3dsRequiredDataMissing') {
            wolkShowInlineError('Error 3DS: faltan datos requeridos para la verificacion. Intente nuevamente o use otra tarjeta.');
        } else {
            wolkShowInlineError(uiKey);
        }
        jQuery('#close-modal').show();

        create_error_log({
            details: e,
            type: e.type || 'Gateway Error'
        });
    }
});

Cardinal.configure({
    logging: {
        level: "verbose"
    },
    timeout: 8000
});

function request_auth(options) {
    try {
        // [OK] Siempre reiniciar UI a "en proceso" al iniciar un nuevo intento
        // Evita que quede la X roja de un error anterior al reintentar dentro de la misma pagina.
        try {
            wolkShowProcessing();
            const t = document.getElementById('responseTitle');
            if (t) t.style.display = '';
            const icon = document.getElementById('responseIcon');
            if (icon) {
                icon.style.display = 'none';
                icon.classList.remove('is-error');
                icon.innerHTML = '';
            }
        } catch (_) {}

        // Ocultar el modal y mostrar el loader
        jQuery('#close-modal').hide();  // Asegurarse de ocultar el boton antes de empezar
        document.getElementsByTagName('authBox')[0].style.display = 'flex';
        document.getElementById('loader').style.display = 'block';

        // Verificar si la interfaz ya esta montada y desmontarla si es necesario
        if (threeDSecureInterface && threeDSecureInterface.isMounted()) {
            safeUnmount(threeDSecureInterface);
        }

        // Crear una nueva instancia del UI de 3DS
        threeDSecureInterface = threeDS.createUI(options);
        threeDSecureInterface.start('modal');

        // Manejo del evento 'challenge'
        threeDSecureInterface.on('challenge', function (e) {
            console.log('=== 3DS CHALLENGE INICIADO ===');
            
            // 1) Ocultar UI interna (deja solo el iframe del challenge)
            var els = ['loader', 'responseTitle', 'response', 'responseIcon', 'close-modal'];
            els.forEach(function (id) {
                var el = document.getElementById(id);
                if (el) el.style.cssText = 'display:none!important';
            });
            
            try { jQuery('#detail-content').hide(); } catch (_) {}
            try { jQuery('.button-container').hide(); } catch (_) {}
            
            // 2) Activar modo 3DS - CSS maneja TODO el layout
            var authBoxEl = document.getElementsByTagName('authBox')[0];
            if (authBoxEl) {
                // LIMPIAR inline styles previos
                authBoxEl.removeAttribute('style');
                // Activar clase
                authBoxEl.classList.add('wolk-3ds-active');
            }
            
            // 3) Ajustar iframe - CSS maneja 100% height
            function applyIframeSize() {
                var iframe =
                    document.querySelector('authBox modal iframe') ||
                    document.querySelector('authBox iframe') ||
                    document.querySelector('modal iframe');
                
                if (!iframe) return false;
                
                // Solo verificamos que existe - CSS maneja el resto
                return true;
            }
            
            var tries = 0;
            var maxTries = 200;
            var t = setInterval(function () {
                tries++;
                if (applyIframeSize() || tries >= maxTries) clearInterval(t);
            }, 50);
            
            function onResize3DS() {
                var ab = document.getElementsByTagName('authBox')[0];
                if (!ab || !ab.classList.contains('wolk-3ds-active')) return;
                applyIframeSize();
            }
            window.addEventListener('resize', onResize3DS, { passive: true });
        });

        // Manejo del evento 'complete'
        threeDSecureInterface.on('complete', function (e) {
            try {
                console.log('=== 3DS COMPLETE ===', e);
                
                // Quitar clase 3DS active del authBox y LIMPIAR estilos
                var authBoxEl = document.getElementsByTagName('authBox')[0];
                var isDesktop = window.innerWidth >= 600;
                if (authBoxEl) {
                    authBoxEl.classList.remove('wolk-3ds-active');
                    // Restaurar estilos originales - LIMPIAR cssText primero
                    authBoxEl.style.cssText = '';
                    authBoxEl.style.display = 'flex';
                    authBoxEl.style.alignItems = isDesktop ? 'center' : 'flex-end';
                    authBoxEl.style.justifyContent = 'center';
                    authBoxEl.style.background = 'rgba(0,0,0,0.6)';
                    authBoxEl.style.padding = isDesktop ? '24px' : '0';
                    authBoxEl.style.position = 'fixed';
                    authBoxEl.style.top = '0';
                    authBoxEl.style.left = '0';
                    authBoxEl.style.right = '0';
                    authBoxEl.style.bottom = '0';
                    authBoxEl.style.width = '100%';
                    authBoxEl.style.height = '100%';
                    authBoxEl.style.zIndex = '999999';
                }
                
                // Restaurar estilos del modal - LIMPIAR cssText primero
                var modalEl = document.getElementsByTagName('modal')[0];
                if (modalEl) {
                    modalEl.style.cssText = '';
                    modalEl.style.position = 'relative';
                    modalEl.style.width = '100%';
                    modalEl.style.maxWidth = isDesktop ? '440px' : '100%';
                    modalEl.style.height = '';
                    modalEl.style.minHeight = '';
                    modalEl.style.maxHeight = isDesktop ? '85vh' : '92vh';
                    modalEl.style.borderRadius = isDesktop ? '20px' : '24px 24px 0 0';
                    modalEl.style.margin = '0';
                    modalEl.style.padding = '0';
                    modalEl.style.overflow = 'hidden';
                    modalEl.style.boxShadow = isDesktop ? '0 12px 40px rgba(0,0,0,0.2)' : '0 -10px 50px rgba(0,0,0,0.3)';
                    modalEl.style.background = '#ffffff';
                }
                
                // Restaurar visibilidad del modal
                jQuery('#detail-content').css('display', 'flex');
                jQuery('.button-container').css('display', 'block');
                document.getElementById('loader').style.display = 'none';
                document.getElementById('responseTitle').style.display = 'block';
                document.getElementById('response').style.display = 'block';
                safeUnmount(threeDSecureInterface);

                // Verificacion de los codigos de transaccion
                if (e && e.code) {
                    if (e.code === 'TRANSACTION_STATUS_N' || e.code === 'TRANSACTION_STATUS_U' || e.code === 'TRANSACTION_STATUS_R') {
                        create_error_log({
                            details: e,
                            type: '3DSecure - Complete - ' + e.code
                        });
                        wolkShowInlineError(e.code);
                        wolkAddOrderNote('Autenticacion 3DS fallida: ' + _lang(e.code));
                        jQuery('#close-modal').show(); // Mostrar el boton solo si la transaccion no es exitosa
                    }
                } else if (e.type === 'gatewayError') {
                    create_error_log({
                        details: e,
                        type: '3DSecure - Complete - Gateway Error'
                    });
                    wolkShowInlineError('gatewayError');
                    wolkAddOrderNote('Error 3DS: ' + _lang('gatewayError'));
                    jQuery('#close-modal').show(); // Mostrar el boton en caso de error del gateway
                }

                // Verificar autenticacion del titular de la tarjeta
                if (e && (e.cardHolderAuth === 'verified' || e.cardHolderAuth === 'attempted')) {
                    sendResponse({
                        ...options,
                        cavv: e.cavv,
                        xid: e.xid,
                        eci: e.eci,
                        cardHolderAuth: e.cardHolderAuth,
                        threeDsVersion: e.threeDsVersion,
                        directoryServerId: e.directoryServerId,
                        cardHolderInfo: e.cardHolderInfo,
                    }, 'verified');
                    // Mantener siempre el mismo estado "en proceso"; no mostrar pantallas intermedias de "exito".
                    wolkShowProcessing();
                    // Ocultar el boton: la transaccion aun no ha terminado
                    jQuery('#close-modal').hide();
                }

                // Manejar el caso de cardHolderAuth nulo
                if (e && e.cardHolderAuth == null) {
                    create_error_log({
                        details: e,
                        type: '3DSecure - Complete - Null CardHolderAuth'
                    });
                    wolkShowInlineError('NullcardHolderAuth');
                    wolkAddOrderNote('Pago rechazado: ' + _lang('NullcardHolderAuth'));
                    jQuery('#close-modal').show(); // Mostrar el boton en caso de error
                }
            } catch (error) {
                console.error('Error en evento complete:', error);
                create_error_log({
                    details: { error: error.message, stack: error.stack },
                    type: '3DSecure - Complete - Exception'
                });
                jQuery('#close-modal').show(); // Mostrar el boton en caso de error inesperado
            }
        });

        // Manejo del evento 'failure'
        threeDSecureInterface.on('failure', function (e) {
            console.log('3d failure', e);
            create_error_log({
                details: e,
                type: '3DSecure - Failure'
            });
            // Restaurar estilos del authBox
            var authBoxEl = document.getElementsByTagName('authBox')[0];
            var isDesktop = window.innerWidth >= 600;
            if (authBoxEl) {
                authBoxEl.style.alignItems = isDesktop ? 'center' : 'flex-end';
                authBoxEl.style.justifyContent = 'center';
                authBoxEl.style.background = 'rgba(0,0,0,0.6)';
                authBoxEl.style.padding = isDesktop ? '24px' : '0';
            }
            
            // Restaurar estilos del modal
            var modalEl = document.getElementsByTagName('modal')[0];
            if (modalEl) {
                modalEl.style.position = 'relative';
                modalEl.style.width = '100%';
                modalEl.style.maxWidth = isDesktop ? '440px' : '100%';
                modalEl.style.height = '';
                modalEl.style.minHeight = '';
                modalEl.style.maxHeight = isDesktop ? '85vh' : '92vh';
                modalEl.style.borderRadius = isDesktop ? '20px' : '24px 24px 0 0';
                modalEl.style.margin = '0';
                modalEl.style.animation = '';
                modalEl.style.boxShadow = isDesktop ? '0 12px 40px rgba(0,0,0,0.2)' : '0 -10px 50px rgba(0,0,0,0.3)';
            }
            
            // Restaurar visibilidad del modal
            jQuery('#detail-content').css('display', 'flex');
            jQuery('.button-container').css('display', 'block');
            document.getElementById('loader').style.display = 'none';
            document.getElementById('responseTitle').style.display = 'block';
            document.getElementById('response').style.display = 'block';
            safeUnmount(threeDSecureInterface);
            if (e && (e.code === 'TRANSACTION_STATUS_N' || e.code === 'TRANSACTION_STATUS_U' || e.code === 'TRANSACTION_STATUS_R')) {
                document.getElementById('response').innerHTML = _lang(e.code);
            }
            if (e && e.type === 'gatewayError') {
                document.getElementById('response').innerHTML = _lang(e.type);
            }
            jQuery('#close-modal').show(); // Mostrar el boton al finalizar con fallo
        });

        // Manejo del evento 'error'
        threeDSecureInterface.on('error', function (e) {
            console.log('3d Error', e);
            create_error_log({
                details: e,
                type: '3DSecure - Error'
            });
            // Restaurar estilos del authBox
            var authBoxEl = document.getElementsByTagName('authBox')[0];
            var isDesktop = window.innerWidth >= 600;
            if (authBoxEl) {
                authBoxEl.style.alignItems = isDesktop ? 'center' : 'flex-end';
                authBoxEl.style.justifyContent = 'center';
                authBoxEl.style.background = 'rgba(0,0,0,0.6)';
                authBoxEl.style.padding = isDesktop ? '24px' : '0';
            }
            
            // Restaurar estilos del modal
            var modalEl = document.getElementsByTagName('modal')[0];
            if (modalEl) {
                modalEl.style.position = 'relative';
                modalEl.style.width = '100%';
                modalEl.style.maxWidth = isDesktop ? '440px' : '100%';
                modalEl.style.height = '';
                modalEl.style.minHeight = '';
                modalEl.style.maxHeight = isDesktop ? '85vh' : '92vh';
                modalEl.style.borderRadius = isDesktop ? '20px' : '24px 24px 0 0';
                modalEl.style.margin = '0';
                modalEl.style.animation = '';
                modalEl.style.boxShadow = isDesktop ? '0 12px 40px rgba(0,0,0,0.2)' : '0 -10px 50px rgba(0,0,0,0.3)';
            }
            
            // Restaurar visibilidad del modal
            jQuery('#detail-content').css('display', 'flex');
            jQuery('.button-container').css('display', 'block');
            document.getElementById('loader').style.display = 'none';
            document.getElementById('responseTitle').style.display = 'block';
            document.getElementById('response').style.display = 'block';
            safeUnmount(threeDSecureInterface);
            document.getElementById('response').innerHTML = _lang('generalError');
            jQuery('#close-modal').show(); // Mostrar el boton al finalizar con error
        });

    } catch (err) {
        console.log(err);
        create_error_log({
            details: { error: err.message, stack: err.stack },
            type: 'request_auth - Exception'
        });
        jQuery('#close-modal').show(); // Mostrar el boton en caso de excepcion general
    }
}

function sendResponse(options, status) {
    if (status == "verified") {
        precess_pay_now(options);
    }
}

function _lang(text) {
    // langu ya es un objeto JS, no necesita JSON.parse
    let lang = langu;
    if (!lang || typeof lang !== 'object') {
        return text;
    }
    text = (text || '').toString().trim();
    if (!sys_lang) {
        return text;
    } else if (sys_lang == 'en') {
        if (lang['en'] && (text in lang['en'])) {
            return lang['en'][text];
        } else {
            return text;
        }
    } else if (sys_lang == 'es') {
        if (lang['es'] && (text in lang['es'])) {
            return lang['es'][text];
        } else {
            return text;
        }
    }
    return text;
}

const precess_pay_now = (values) => {
    let cc = jQuery("#wolk_ccNo").val()
    let customer_ip = jQuery("#customer_ip").val()
    let wolk_first_name = jQuery("#wolk_first_name").val()
    let wolk_last_name = jQuery("#wolk_last_name").val()
    let wolk_email = jQuery("#wolk_email").val()
    let cardholder = jQuery("#wolk_cardholder").val()
    let wolk_month = jQuery("#wolk_month").val()
    let wolk_year = jQuery("#wolk_year").val()
    let wolk_cvc = jQuery("#wolk_cvc").val()

    cc = cc.replaceAll(' ', '');

    var data = new FormData();
    let rawdata = {
        'action': 'wolk_submit_payment_request',
        // [OK] IMPORTANT: required so PHP can store the BAC/NMI response in the WC order
        // and later allow `wolk_complete_order()` to redirect to the thank-you page.
        // Without this, `_wolk_response_code` is never saved and complete_order returns FAIL.
        'order_id': (typeof order_id !== 'undefined' ? order_id : ''),
        // SEGURIDAD PCI: security_key REMOVIDA - se agrega en backend
        'type': 'sale',
        'amount': total,
        'ccnumber': cc,
        'ccexp': wolk_month + '/' + wolk_year,
        'cvv': wolk_cvc,
        'firstname': wolk_first_name,
        'lastname': wolk_last_name,
        // Titular real de la tarjeta (puede diferir del comprador/cliente)
        // Se usa para auditoria en notas del pedido.
        'cc_name': (cardholder || ''),
        'email': wolk_email,
        'ipaddress': customer_ip,
        currency: currency_code,
        country: country,
        xid: values.xid == null ? null : values.xid
    };

    // Agregar processor solo si existe y es valido
    if (values.processor !== undefined && values.processor !== null) {
        rawdata.processor = values.processor;
    }

    // Agregar el resto de propiedades de values (excepto processor y xid)
    for (let key in values) {
        if (key !== 'processor' && key !== 'xid') {
            rawdata[key] = values[key];
        }
    }

    for (var key in rawdata) {
        data.append(key, rawdata[key])
    }

    // Incluir el canal de dispositivo usado (para trazabilidad en notas y auditoria)
    // NMI/Gateway.js usa 'Browser' para la pagina de pago.
    data.append('deviceChannel', 'Browser');

    data.append('nonce', ajax_var.nonce)

    var config = {
        method: 'POST',
        body: data
    };

    fetch(ajax_var.url, config)
        .then(response => response.text())
        .then(async function (response) {
            let res = await queryStringToJSON(response)
            if (res.response == 1) {
                wolkShowProcessing();
                // ! Importante: no mostramos "Aprobada" aqui; esperamos confirmacion de wolk_complete_order
                //document.getElementById('responseTitle').innerHTML = _lang('Transaction Approved')
                //document.getElementById('response').innerHTML =  _lang(Transaction ID = ${res.transactionid})
                setTimeout(() => {
                    var formData = new FormData();
                    formData.append('action', 'wolk_complete_order');
                    formData.append('order_id', order_id);
                    // Trazabilidad: canal de dispositivo
                    formData.append('deviceChannel', 'Browser');
                    formData.append('nonce', ajax_var.nonce)

                    fetch(ajax_var.url,
                        {
                            method: 'POST',
                            body: formData,
                        })
                        .then(res => res.json())
                        .then(res => {
                            if (res.result == "success") {
                                // [OK] Ya WooCommerce confirmo; si quisieras mostrar "aprobada" 0.5s antes de redirigir, aqui seria el lugar seguro.
                                // document.getElementById('responseTitle').innerHTML = _lang('Transaction Approved');
                                window.location = res.redirect;
                            } else {
                                // [ERROR] Si el backend dice FAIL, mostramos el mensaje real y registramos para auditoria.
                                wolkShowInlineError(res.message || _lang('Error'));
                                wolkAddOrderNote('Error al finalizar pedido: ' + (res.message || 'No se pudo completar.'));
                                create_error_log({ details: res, type: 'Complete Order - Fail' });
                            }
                        })
                        .catch(err => console.log(err));
                }, 1000)
            } else if (res.response == 2) {
                create_error_log({
                    details: res,
                    type: 'Payment Proceed - Declined'
                });
                jQuery('#detail-content').show()
                safeUnmount(threeDSecureInterface)
                jQuery('#close-modal').show();
                const uiKey = wolkResolveUiError(res);
                wolkShowInlineError(uiKey);
                // Nota de pedido: ya se registra en backend (evita duplicados y evita genericos cuando existe un motivo especifico).
            } else if (res.response == 3) {
                create_error_log({
                    details: res,
                    type: 'Payment Proceed - Error'
                });
                jQuery('#detail-content').show()
                safeUnmount(threeDSecureInterface)
                jQuery('#close-modal').show();
                const uiKey = wolkResolveUiError(res);
                wolkShowInlineError(uiKey);
                // Nota de pedido: ya se registra en backend (evita duplicados).
            }
            else {
                create_error_log({
                    details: res,
                    type: 'Payment Proceed - Unknown'
                });
                jQuery('#detail-content').show()
                safeUnmount(threeDSecureInterface)
                jQuery('#close-modal').show();
                const uiKey = wolkResolveUiError(res);
                wolkShowInlineError(uiKey);
                // Nota de pedido: ya se registra en backend (evita duplicados).
            }
        })
        .catch(function (error) {
            console.log(error);
            create_error_log({
                details: { error: error.message, stack: error.stack },
                type: 'Payment Proceed - Fetch Exception'
            });
            wolkShowInlineError('generalError');
            wolkAddOrderNote('Error de pago: ' + _lang('generalError'));
        });
}

function queryStringToJSON(queryString) {
    // Algunos entornos devuelven el querystring HTML-escapado ("&amp;" en vez de "&").
    // Eso rompe el parser y deja llaves como "amp;responsetext".
    // Normalizamos para que la UI pueda detectar el motivo real (fondos, cvv, etc.).
    if (typeof queryString === 'string') {
        queryString = queryString.replaceAll('&amp;', '&');
    }

    if (queryString.indexOf('?') > -1) {
        queryString = queryString.split('?')[1];
    }
    var pairs = queryString.split('&');
    var result = {};
    pairs.forEach(function (pair) {
        pair = pair.split('=');
        var k = pair[0] || '';
        // Si quedo "amp;foo" por escapado parcial, lo corregimos.
        if (k.startsWith('amp;')) k = k.substring(4);
        result[k] = decodeURIComponent(pair[1] || '');
    });
    return result;
}

function creditCardType(cc) {
    if (cc.substring(0, 2) == "34" || cc.substring(0, 2) == "37") {
        return "AMEX"
    }
    else {
        return "other"
    }
}

function show_form_error() {
    jQuery('#purchase').remove()
    jQuery('.form-error').text(_lang("payment-form-not-available"))
    jQuery(".form-error").css({
        color: "red"
    })
}

// ---- NUEVA FUNCION ANADIDA AQUI ----
function show_3ds_inactive_error(errorDetails){
    // Oculta el boton de compra
    jQuery('#purchase').hide(); // Cambiado de .remove() a .hide() para poder mostrarlo de nuevo si se resuelve

    // Crea el mensaje combinado: Mensaje amigable + Detalle tecnico
    // Usamos _lang para el mensaje amigable
    let mensajeAmigable = _lang("3DSecure is inactive on your account. Please contact your bank agent to activate 3DS, and send this error evidence.");

    // Concatenamos el detalle tecnico
    let mensajeCompleto = mensajeAmigable + " " + (errorDetails.message || 'N/A');

    // Muestra el mensaje combinado en el contenedor original
    jQuery('.form-error').text(mensajeCompleto).css({ color:"red" });

    // Registra el detalle tecnico en consola y log (solo para diagnostico interno)
    console.error("3DSecure Inactivo - Detalles:", errorDetails);
    
    create_error_log({
        details: errorDetails,
        type: '3DSecure-Inactive'
    });
}
// ---- FIN NUEVA FUNCION ----

jQuery(document).ready(function (jQuery) {
    // Deshabilitar el boton de pago si la licencia esta inactiva
    if (typeof wolk_license_active !== 'undefined' && !wolk_license_active) {
        jQuery('#purchase').prop('disabled', true);
        // NO modificar el mensaje de .form-error aqui, solo deshabilitar campos
        jQuery('#request-payment input, #request-payment select').prop('disabled', true);
        console.error('[WOLK] Formulario deshabilitado por licencia inactiva.');
    }

    if (typeof wolk_license_timeout !== 'undefined' && wolk_license_timeout) {
        console.warn('[WOLK] Timeout de licencia: el pago esta permitido temporalmente.');
    }

    // ---- Validacion de intentos fallidos ----
    let validationAttempts = 0;
    const maxAttempts = 6; // PCI-DSS recomendado

    function handleValidationFail() {
        validationAttempts++;
        console.warn('[WOLK] Intento de validacion fallido #' + validationAttempts);
        // Solo bloquear despues de 6 intentos (proteccion contra bots)
        if (validationAttempts >= maxAttempts) {
            jQuery('#purchase').prop('disabled', true);
            jQuery('.form-error').text(_lang('validation_attempts_exceeded'));
            console.error('[WOLK] Formulario bloqueado por seguridad.');
        }
    }

    let encrypted_processor_id = null;

    // Detectar cambio en el combo box para el plan de cuotas
    jQuery('#installment-plan').on('change', function () {
        let selectedValue = jQuery(this).val();
        
        if (selectedValue === 'single_payment') {
            // Para pago unico, usar null o un valor especifico
            encrypted_processor_id = 'single_payment';
        } else {
            encrypted_processor_id = selectedValue;
            console.log("Processor seleccionado:", encrypted_processor_id);
        }
    });

    // Limpiar errores al escribir
    jQuery("#wolk_first_name").keyup(() => {
        jQuery(".first_name-error").text("")
    })

    jQuery("#wolk_last_name").keyup(() => {
        jQuery(".last_name-error").text("")
    })

    jQuery("#wolk_email").keyup(() => {
        jQuery(".email-error").text("")
    })

    jQuery("#wolk_ccNo").keyup(() => {
        jQuery(".card-error").text("")
    })

    jQuery("#wolk_cardholder").keyup(() => {
        jQuery(".holder-error").text("")
    })

    jQuery("#wolk_month").change(() => {
        jQuery(".expiry-error").text("")
    })

    jQuery("#wolk_year").change(() => {
        jQuery(".expiry-error").text("")
    })

    jQuery("#wolk_cvc").keyup(() => {
        jQuery(".cvc-error").text("")
    })

    jQuery("#request-payment").submit((e) => {
        e.preventDefault();
        
        // Obtener valores
        let customer_ip = jQuery("#customer_ip").val();
        let cc = jQuery("#wolk_ccNo").val();
        let cardholder = jQuery("#wolk_cardholder").val();
        let wolk_month = jQuery("#wolk_month").val();
        let wolk_year = jQuery("#wolk_year").val();
        let wolk_cvc = jQuery("#wolk_cvc").val();
        let wolk_first_name = jQuery("#wolk_first_name").val();
        let wolk_last_name = jQuery("#wolk_last_name").val();
        let wolk_email = wolkSanitizeEmail(jQuery("#wolk_email").val());
let card_type = creditCardType(cc);
        cc = cc.replaceAll(' ', '');
        
        // ============ VALIDACION 1: Plan seleccionado ============
        let encrypted_processor_id = jQuery('#installment-plan').val();
        
        if (!encrypted_processor_id || encrypted_processor_id === '') {
            wolkShowAlert("Por favor selecciona un plan de pago", "warning");
            return false;
        }
        
        // ============ VALIDACION 2: Order ID existe ============
        if (typeof order_id === 'undefined' || !order_id) {
            console.error("order_id no esta definido.");
            wolkShowAlert("No se encontro la orden", "error");
            return false;
        }
        
        // ============ FUNCION DE VALIDACION REUTILIZABLE ============
        function validarFormulario() {
            if (wolk_first_name == "" || wolk_first_name.length < 2) {
                jQuery(".first_name-error").text(_lang("invalid first name.")).css({ color: "red" });
                jQuery("#wolk_first_name").focus();
                return false;
            }
            if (wolk_last_name == "" || wolk_last_name.length < 2) {
                jQuery(".last_name-error").text(_lang("invalid last name.")).css({ color: "red" });
                jQuery("#wolk_last_name").focus();
                return false;
            }
            if (card_type == "AMEX" && cc.length < 15) {
                jQuery(".card-error").text(_lang("invalid card number.")).css({ color: "red" });
                jQuery("#wolk_ccNo").focus();
                return false;
            }
            if (card_type !== "AMEX" && cc.length < 16) {
                jQuery(".card-error").text(_lang("invalid card number.")).css({ color: "red" });
                jQuery("#wolk_ccNo").focus();
                return false;
            }
            if (cardholder == "" || cardholder.length < 3) {
                jQuery(".holder-error").text(_lang("invalid card holder.")).css({ color: "red" });
                jQuery("#wolk_cardholder").focus();
                return false;
            }
            if (!wolkIsValidEmail(wolk_email)) {
                jQuery(".email-error").text(_lang("invalid email.")).css({ color: "red" });
                jQuery("#wolk_email").focus();
                return false;
            }
            if (wolk_month == "00" || wolk_year == "00") {
                jQuery(".expiry-error").text(_lang("invalid expiry date.")).css({ color: "red" });
                return false;
            }
            if (wolk_cvc.length <= 2 || wolk_cvc == "" || wolk_cvc == " ") {
                jQuery(".cvc-error").text(_lang("invalid cvc.")).css({ color: "red" });
                jQuery("#wolk_cvc").focus();
                return false;
            }
            return true;
        }
        
        // ============ VALIDACION 3: Datos del formulario ============
        if (!validarFormulario()) {
            handleValidationFail();
            return false;
        }
        
        // ============ CASO 1: PAGO UNICO ============
        if (encrypted_processor_id === 'single_payment') {
            console.log('Pago Unico - Sin processor_id');
            
            const options = {
                cardNumber: cc,
                cardExpMonth: wolk_month,
                cardExpYear: wolk_year,
                currency: currency_code,
                country: country,
                amount: total,
                firstName: wolk_first_name,
                lastName: wolk_last_name,
                email: wolk_email,
                // Manual device data for 3DS2 browser channel
                ...wolkBuild3DSDeviceData()
                // <- SIN processor
                };
                request_auth(options);
            return false;
        }
        
        // ============ CASO 2: MINICUOTAS O TASA0 ============
        // Necesita desencriptar el processor_id
        jQuery.ajax({
            url: ajax_var.url,
            method: 'POST',
            data: {
                action: 'decrypt_processor_id',
                encrypted_processor_id: encrypted_processor_id,
                order_id: order_id,
            },
            dataType: 'json',
            timeout: 5000,
            success: function (response) {
                console.log('Decrypt response:', response);
                
                // [OK] Validar respuesta
                if (!response.success) {
                    console.error("Error en desencriptacion:", response.data?.message || 'Desconocido');
                    wolkShowAlert("Error al procesar el plan de cuotas. Por favor, intentalo de nuevo.", "error");
                    return false;
                }
                
                // [OK] Validar que processor_id existe
                if (!response.data || response.data.processor_id === undefined) {
                    console.error("processor_id no devuelto en respuesta");
                    wolkShowAlert("No se pudo obtener el ID del plan. Intentalo de nuevo.", "error");
                    return false;
                }
                
                let processor_id = response.data.processor_id;
                console.log('Processor ID desencriptado:', processor_id);
                
                // Procesar pago con processor_id
                const options = {
                    cardNumber: cc,
                    cardExpMonth: wolk_month,
                    cardExpYear: wolk_year,
                    currency: currency_code,
                    country: country,
                    amount: total,
                    firstName: wolk_first_name,
                    lastName: wolk_last_name,
                    email: wolk_email,
                    processor: processor_id,
                    // Manual device data for 3DS2 browser channel
                    ...wolkBuild3DSDeviceData()
                };
                request_auth(options);
            },
            error: function (xhr, status, error) {
                console.error('AJAX Error:', status, error, xhr.responseText);
                wolkShowAlert("Error de conexion. Verifica tu conexion a internet e intentalo de nuevo.", "error");
            }
        });
        
        return false;
    });

    jQuery("#close-modal").click(() => {
        jQuery("authBox").hide()
        jQuery("#loader").hide()
        // Resetear UI para permitir reintento limpio (sin X roja pegada)
        jQuery("#responseTitle").show().html("")
        jQuery("#response").html("")
        jQuery("#responseIcon").hide().removeClass('is-error').html('')
        jQuery('#close-modal').hide();
        jQuery("#wolk_ccNo").val('')
        jQuery("#wolk_cardholder").val('')
        jQuery("#wolk_month").val('00')
        jQuery("#wolk_year").val('00')
        jQuery("#wolk_cvc").val('')
        jQuery("#wolk_first_name").val('')
        jQuery("#wolk_last_name").val('')
        jQuery("#wolk_email").val('')
        jQuery("#processor_id").val('')
        if (threeDSecureInterface) {
            safeUnmount(threeDSecureInterface)
        }
    })
})

// Auto-rellena el nombre del titular de la tarjeta basandose en el nombre y apellido
jQuery(document).ready(function ($) {
    $('#wolk_first_name, #wolk_last_name').on('input', function () {
        let firstName = $('#wolk_first_name').val();
        let lastName = $('#wolk_last_name').val();
        $('#wolk_cardholder').val(firstName + ' ' + lastName);
    });
});

// ============================================================
// FUNCION CORREGIDA: create_error_log
// ============================================================
/**
 * Guarda logs de errores en la base de datos
 * @param {Object} logData - Objeto con la estructura: { details, type, cc_last_6?, cc_name? }
 */
function create_error_log(logData) {
    try {
        // Extraer datos de la tarjeta del DOM si no se proporcionan
        let cc_last_6 = logData.cc_last_6 || '';
        let cc_name = logData.cc_name || '';
        
        // Si no hay datos, intentar obtenerlos del formulario
        if (!cc_last_6 || !cc_name) {
            let cc = jQuery("#wolk_ccNo").val() || '';
            let cardholder = jQuery("#wolk_cardholder").val() || '';
            
            // Limpiar numero de tarjeta y obtener ultimos 6 digitos
            if (cc) {
                cc = cc.replace(/\s+/g, ''); // Eliminar espacios
                if (cc.length >= 6) {
                    cc_last_6 = cc.slice(-6); // Ultimos 6 digitos
                } else {
                    cc_last_6 = 'INVALID_LENGTH';
                }
            } else {
                cc_last_6 = 'NOT_AVAILABLE';
            }
            
            cc_name = cardholder || 'NOT_AVAILABLE';
        }
        
        // Log en consola para debugging
        console.log('[WOLK] Guardando error log:', {
            order_id: typeof order_id !== 'undefined' ? order_id : 'NO_ORDER_ID',
            cc_last_6: cc_last_6,
            cc_name: cc_name,
            type: logData.type || 'unknown'
        });
        
        jQuery.ajax({
            url: ajax_var.url,
            method: 'POST',
            data: {
                action: 'wolk_submit_error_logs',
                order_id: typeof order_id !== 'undefined' ? order_id : 'NO_ORDER_ID',
                cc_last_6: cc_last_6,
                cc_name: cc_name,
                details: JSON.stringify(logData.details || {}),
                type: logData.type || 'unknown',
                nonce: ajax_var.nonce
            },
            dataType: 'json',
            success: function (response) {
                console.log('[WOLK] Log guardado exitosamente:', response);
            },
            error: function (xhr, status, error) {
                console.error('[WOLK] Error al guardar log:', {
                    status: status,
                    error: error,
                    response: xhr.responseText
                });
            }
        });
    } catch (e) {
        console.error('[WOLK] Excepcion en create_error_log:', e.message, e.stack);
    }
}