<?php
/* @wordpress-plugin
 * Plugin Name: Wolk Payment Cardinal
 * Plugin URI: https://wolksoftcr.com/plug-in-payment-gateway
 * Description: <p>Presenting our 3D Secure-based secure payment plugin: A robust solution that ensures the security of your users' transactions. Our plugin fully complies with international payment security protocols, providing you with peace of mind. Installation is simple, and for a limited time, you can get one year of enhanced security for just $75 or two years for $100. Plus, we offer 24/7 support to assist you with any questions or concerns regarding the plugin.</p>
 * Version: 1.7.0
 * Author: WolkSoftware
 * Author URI: https://wolksoftcr.com/
 * Text Domain: wolk-payment-gateway
 * Domain Path: /languages
 * License: GPL-2.0+
 * License URI: http://www.gnu.org/licenses/gpl-2.0.txt
 */

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly 

// ============================================================================
// SISTEMA DE AUTO-ACTUALIZACIÓN - CONSTANTES
// ============================================================================
define('WOLK_PAYMENT_VERSION', '1.7.0');
define('WOLK_PAYMENT_LAST_UPDATE', '2026-01-21');
define('WOLK_PAYMENT_UPDATE_CHECK_URL', 'https://wolksoftcr.com/wp-content/uploads/wolk-latest-version.json');
// ============================================================================

require 'auth/class-wolk-license-auth.php';
// table name

$active_plugins = apply_filters('active_plugins', get_option('active_plugins'));
if(wolk_payment_is_woocommerce_active()){
    add_filter('woocommerce_payment_gateways', 'wolk_init_wolk_payment_gateway');
    function wolk_init_wolk_payment_gateway( $gateways ){
        $gateways[] = 'Wolk_Payment_Gateway';
        return $gateways;
    }

    add_action('plugins_loaded', 'wolk_include_wolk_payment_gateway');
    function wolk_include_wolk_payment_gateway(){
        require 'class-wolk-payment-gateway.php';
    }

    add_action( 'plugins_loaded', 'wolk_payment_load_plugin_textdomain' );
    function wolk_payment_load_plugin_textdomain() {
        load_plugin_textdomain( 'wolk-payment-gateway', false, basename( dirname( __FILE__ ) ) . '/languages/' );
    }

    // Hook the custom function to the 'woocommerce_blocks_loaded' action
    add_action( 'woocommerce_blocks_loaded', 'register_order_approval_payment_method_type' );

    /**
     * Custom function to register a payment method type

     */
    function register_order_approval_payment_method_type() {
        // Check if the required class exists
        if ( ! class_exists( 'Automattic\WooCommerce\Blocks\Payments\Integrations\AbstractPaymentMethodType' ) ) {
            return;
        }

        // Include the custom Blocks Checkout class
        require_once 'class-block.php';

        // Hook the registration function to the 'woocommerce_blocks_payment_method_type_registration' action
        add_action(
            'woocommerce_blocks_payment_method_type_registration',
            function( Automattic\WooCommerce\Blocks\Payments\PaymentMethodRegistry $payment_method_registry ) {
                // Register an instance of Wolk_Payment_Gateway_Blocks
                $payment_method_registry->register( new Wolk_Payment_Gateway_Blocks() );
            }
        );
    }
}
function wolk_enqueue_wolk_custom_admin_script_test() {
    wp_enqueue_style( 'datatable_style', plugin_dir_url(__FILE__).'/assets/dist/datatables/datatables.css' , false );
    wp_enqueue_script( 'datatables', plugin_dir_url(__FILE__).'/assets/dist/datatables/datatables.min.js', array( 'jquery' ) );

    /// Paypal Buttons
    wp_register_script( 'paypal-buttons', '//www.paypal.com/sdk/js?client-id=AWMDsfc9toExRdurz4LZy6T_XIjTLUsWtB2jEy1wvaMTrcHjDcWaXR5t7T9kiT3a5NN1PxQvDHhG4cmf&vault=true&intent=subscription',array(),
        null,
        false );
    wp_enqueue_script('paypal-buttons');
    wp_enqueue_style( 'my_style', plugin_dir_url(__FILE__).'/assets/style.css' , false );

    // wp_localize_script('myplugin-ajax', 'ajax_var', array(
    // 	'url' => admin_url('admin-ajax.php'),
    // 	'nonce' => wp_create_nonce('ajax-nonce')
    // ));
}

add_action( 'admin_enqueue_scripts', 'wolk_enqueue_wolk_custom_admin_script_test' );

function wolk_enqueue_wolk_custom_script() {
    // SEGURIDAD: Solo cargar en la pagina de pago wolk-payment-page
    // Verificar por URL directamente (mas confiable que is_page)
    $current_url = $_SERVER['REQUEST_URI'];
    
    // Solo cargar si la URL contiene wolk-payment-page
    if (strpos($current_url, 'wolk-payment-page') === false) {
        return;
    }
    
    wp_enqueue_style(
        'payment-proccess-css',
        plugin_dir_url(__FILE__).'/templates/payment-proccess.css',
        array(),
        '3.3.1',
        'all'
    );
    wp_enqueue_script( 'networkmerchants', 'https://secure.networkmerchants.com/js/v1/Gateway.js', array(),'1.0.0', true);
    wp_enqueue_script( 'payment-proccess', plugin_dir_url(__FILE__).'/templates/payment-proccess.js', array('jquery', 'networkmerchants'),'1.7.2', true);
    wp_enqueue_script( 'payment-function', plugin_dir_url(__FILE__).'/assets/function.js', array('jquery'),'1.0.0', true);

    /// AJAX NONCE
    wp_localize_script('payment-proccess', 'ajax_var', array(
        'url' => admin_url('admin-ajax.php'),
        'nonce' => wp_create_nonce('ajax-nonce')
    ));
}

add_action( 'wp_enqueue_scripts', 'wolk_enqueue_wolk_custom_script' );

/**
 * Prevent page cache/optimization issues on dynamic payment pages.
 *
 * Why:
 * - The payment page contains dynamic order context (nonces, 3DS, AJAX).
 * - Some cache/optimization plugins may cache or alter HTML/JS and break the UI.
 *
 * Scope:
 * - Always targets the stable slug: /wolk-payment-page/
 * - Also covers WooCommerce payment endpoints (order-pay / order-received / checkout)
 *
 * Notes:
 * - We do NOT assume any particular cache plugin. We set standard WP signals
 *   (DONOTCACHEPAGE + nocache_headers).
 * - If LiteSpeed Cache is present, we additionally request a no-cache response
 *   using its public action (safe no-op if not installed).
 */
function wolk_payment_disable_cache_on_payment_pages() {
    if ( is_admin() ) {
        return;
    }

    $uri = $_SERVER['REQUEST_URI'] ?? '';

    // Stable across domains/clients.
    $is_wolk_payment_page = ( stripos( $uri, '/wolk-payment-page/' ) !== false );

    // Extra safety for standard WC payment contexts.
    $is_wc_payment_context =
        ( function_exists( 'is_checkout' ) && is_checkout() ) ||
        ( function_exists( 'is_wc_endpoint_url' ) && ( is_wc_endpoint_url( 'order-pay' ) || is_wc_endpoint_url( 'order-received' ) ) );

    if ( ! $is_wolk_payment_page && ! $is_wc_payment_context ) {
        return;
    }

    // Standard cache-bypass signals respected by many cache plugins.
    if ( ! defined( 'DONOTCACHEPAGE' ) ) {
        define( 'DONOTCACHEPAGE', true );
    }
    if ( ! defined( 'DONOTCACHEDB' ) ) {
        define( 'DONOTCACHEDB', true );
    }
    if ( ! defined( 'DONOTCACHEOBJECT' ) ) {
        define( 'DONOTCACHEOBJECT', true );
    }

    // LiteSpeed Cache (optional): request no-cache response if plugin is present.
    if ( defined( 'LSCWP_V' ) ) {
        do_action( 'litespeed_control_set_nocache' );
    }

    // Force no-cache headers (prevents serving stale HTML/JS for payment flow).
    nocache_headers();
    if ( ! headers_sent() ) {
        header( 'Cache-Control: no-store, no-cache, must-revalidate, max-age=0' );
        header( 'Pragma: no-cache' );
        header( 'Expires: Wed, 11 Jan 1984 05:00:00 GMT' );
    }
}
add_action( 'template_redirect', 'wolk_payment_disable_cache_on_payment_pages', 0 );

/**
 * @return bool
 */
function wolk_payment_is_woocommerce_active()
{
    $active_plugins = (array) get_option('active_plugins', array());

    if (is_multisite()) {
        $active_plugins = array_merge($active_plugins, get_site_option('active_sitewide_plugins', array()));
    }

    return in_array('woocommerce/woocommerce.php', $active_plugins) || array_key_exists('woocommerce/woocommerce.php', $active_plugins);
}

add_action( 'before_woocommerce_init', function() {
    if ( class_exists( \Automattic\WooCommerce\Utilities\FeaturesUtil::class ) ) {
        \Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility( 'custom_order_tables', __FILE__, true );
    }
} );

/** Activation hook **/
register_activation_hook(__FILE__, 'wolk_plugin_activation_init');
register_activation_hook(__FILE__, 'wolk_plugin_activation_init_create_table');
function wolk_plugin_activation_init() {

    $post_tite = 'Wolk Payment Page';
    $payment_post = new WP_Query(
        array(
            'post_type'              => 'page',
            'title'                  => $post_tite
        )
    );
    if(empty( $payment_post->post )){
        $payment_proccess_post = array(
            'post_title'    => wp_strip_all_tags($post_tite),
            'post_content'  => '',
            'post_status'   => 'publish',
            'post_author'   => 1,
            'post_type'     => 'page',
        );
        // Insert the post into the database
        $page = wp_insert_post( $payment_proccess_post );
        add_option('wolk_wc_payment_page_id',$page,'','yes');
        add_post_meta($page, '_wp_page_template', 'payment-process.php');
    }



}

function wolk_plugin_activation_init_create_table() {

    global $wpdb;
    $wolk_error_logs_table_name = $wpdb->prefix . 'wolk_error_logs';
    // add error logs table
    $wolk_error_logs_db_version = '1.0.0';
    $charset_collate = $wpdb->get_charset_collate();

    if($wpdb->get_var( $wpdb->prepare("SHOW TABLES LIKE %s", $wolk_error_logs_table_name)) != $wolk_error_logs_table_name )
    {

        // echo $charset_collate;
        // wp_die();

        $sql = "CREATE TABLE `$wolk_error_logs_table_name` (
					`id` int(11) NOT NULL auto_increment,
					`order_id` varchar(256) NOT NULL,
					`cc_last_6` varchar(15) NOT NULL,
					`cc_name` varchar(60) NOT NULL,
					`details` text NOT NULL,
					`type` varchar(50) NOT NULL,
					`created_time` TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
					UNIQUE KEY id (id)
			) ENGINE=MyISAM DEFAULT CHARSET=latin1 AUTO_INCREMENT=1 ;";

        require_once( ABSPATH . 'wp-admin/includes/upgrade.php' );
        dbDelta( $sql );
        add_option( 'wolk_error_logs_db_version', $wolk_error_logs_db_version );
    }

}

function wolk_custom_templates($templates) {

    $templates['payment-process.php'] = 'Payment Process';
    // Add more templates as needed
    return $templates;
}

add_filter('theme_page_templates', 'wolk_custom_templates');


function wolk_assign_custom_template($template) {
    $current_template = get_post_meta(get_the_ID(), '_wp_page_template', true);

    if ($current_template === 'payment-process.php') {
        $template = plugin_dir_path(__FILE__) . 'templates/payment-process.php';
    }
    // Add more conditions for other templates

    return $template;
}
add_filter('page_template', 'wolk_assign_custom_template');



function wolk_complete_order() {

    global $woocommerce;
    $Payment_Gateway = new Wolk_Payment_Gateway();

    $order_id = "";
    if ( ! wp_verify_nonce( $_POST['nonce'], 'ajax-nonce' ) ) {
        die ( 'Invalid Nonce!');
    }
    else{
        if(isset($_POST['order_id'])){
            $order_id = esc_html(sanitize_text_field($_POST['order_id']));
        }
        $order = wc_get_order( $order_id );

        // ==============================================================
        // POLITICA DE ESTADOS (UNICO PUNTO DE CAMBIO DE ESTADO)
        // --------------------------------------------------------------
        // El estado de la orden SOLO se debe cambiar AQUI y SOLO cuando
        // el banco haya confirmado el pago como APROBADO.
        //
        // Para blindarlo (evitar que alguien llame este endpoint a mano,
        // o que se ejecute por error), validamos el meta guardado por
        // `wolk_submit_payment_request()`:
        //   _wolk_response_code == '1'  => APROBADO
        //
        // Si NO esta aprobado, NO tocar estado, NO reducir stock y NO
        // vaciar carrito.
        // ==============================================================
        if ( ! $order ) {
            echo wp_json_encode([
                'result'  => 'fail',
                'message' => 'Orden no encontrada.'
            ]);
            exit();
        }

        $response_code = (string) $order->get_meta( '_wolk_response_code' );
        $transaction_id = (string) $order->get_meta( '_wolk_transaction_id' );

        if ( $response_code !== '1' ) {
            // Nota privada para auditoria (no visible al cliente)
            $order->add_order_note(
                sprintf(
                    'WOLK: Intento de completar orden sin aprobacion del banco. response=%s | transactionid=%s',
                    $response_code ? $response_code : 'N/A',
                    $transaction_id ? $transaction_id : 'N/A'
                ),
                1
            );
            $order->save();

            echo wp_json_encode([
                'result'  => 'fail',
                'message' => 'Pago no aprobado. No se actualiza el estado de la orden.'
            ]);
            exit();
        }

        // Si ya esta en un estado final, evitar reproceso
        if ( $order->has_status( array( 'processing', 'completed', 'cancelled', 'refunded', 'failed' ) ) ) {
            echo wp_json_encode([
                'result'   => 'success',
                'redirect' => $Payment_Gateway->get_return_url( $order )
            ]);
            exit();
        }

        // [OK] UNICO CAMBIO DE ESTADO (DESPUES DE response=1)
		$device_channel = '';
		if (isset($_POST['deviceChannel'])) {
			$device_channel = sanitize_text_field($_POST['deviceChannel']);
		} elseif (isset($_POST['device_channel'])) {
			$device_channel = sanitize_text_field($_POST['device_channel']);
		}
		$device_suffix = $device_channel ? (' | deviceChannel=' . $device_channel) : '';

        $order->update_status(
            $Payment_Gateway->get_order_status(),
            sprintf(
				'Pago aprobado por el banco. response=1 | transactionid=%s%s',
				$transaction_id ? $transaction_id : 'N/A',
				$device_suffix
            )
        );

        // Reduce stock levels (solo con pago aprobado)
        wc_reduce_stock_levels( $order_id );
        if(!empty($_POST[$Payment_Gateway->get_id().'-admin-note'])){
            if(isset($_POST[$Payment_Gateway->get_id().'-admin-note']) && trim($_POST[$Payment_Gateway->get_id().'-admin-note']) !=''){
                $order->add_order_note(esc_html(sanitize_text_field($_POST[ $Payment_Gateway->get_id().'-admin-note'])));
            }
        }

        // Remove cart (solo con pago aprobado)
        $woocommerce->cart->empty_cart();
        // Return thankyou redirect
        echo wp_json_encode([
            'result' => 'success',
            'redirect' => $Payment_Gateway->get_return_url( $order )
        ]);
        exit();
    }
}
add_action('wp_ajax_wolk_complete_order', 'wolk_complete_order' ); // executed when logged in
add_action('wp_ajax_nopriv_wolk_complete_order', 'wolk_complete_order' ); // executed when logged out



function wolk_submit_payment_request() {

    // SEGURIDAD: Verificar nonce PRIMERO
    if ( ! isset($_POST['nonce']) || ! wp_verify_nonce( $_POST['nonce'], 'ajax-nonce' ) ) {
        die ( 'Invalid Nonce!');
    }

    // SEGURIDAD PCI: Obtener security_key desde opciones, NUNCA del POST
    $Payment_Gateway = new Wolk_Payment_Gateway();
    $security_key = $Payment_Gateway->get_option('security_key');
    
    if (empty($security_key)) {
        echo 'response=3&responsetext=' . urlencode('Configuration error: security_key not set');
        wp_die();
    }

    $ipaddress = '';
    if (isset($_SERVER['HTTP_CLIENT_IP'])) {
        $ipaddress = $_SERVER['HTTP_CLIENT_IP'];
    } elseif (isset($_SERVER['HTTP_X_FORWARDED_FOR'])) {
        $ipaddress = $_SERVER['HTTP_X_FORWARDED_FOR'];
    } elseif (isset($_SERVER['HTTP_X_FORWARDED'])) {
        $ipaddress = $_SERVER['HTTP_X_FORWARDED'];
    } elseif (isset($_SERVER['HTTP_X_CLUSTER_CLIENT_IP'])) {
        $ipaddress = $_SERVER['HTTP_X_CLUSTER_CLIENT_IP'];
    } elseif (isset($_SERVER['HTTP_FORWARDED_FOR'])) {
        $ipaddress = $_SERVER['HTTP_FORWARDED_FOR'];
    } elseif (isset($_SERVER['HTTP_FORWARDED'])) {
        $ipaddress = $_SERVER['HTTP_FORWARDED'];
    } elseif (isset($_SERVER['REMOTE_ADDR'])) {
        $ipaddress = $_SERVER['REMOTE_ADDR'];
    } else {
        $ipaddress = 'UNKNOWN';
    }

        $query = "";

        // Obtener order_id
        $order_id = isset($_POST['order_id']) ? sanitize_text_field($_POST['order_id']) : '';
        if ( empty($order_id) && isset($_POST['orderid']) ) {
            $order_id = sanitize_text_field($_POST['orderid']);
        }

        $d = $_POST;

        // SEGURIDAD PCI: Agregar security_key desde backend
        $d['security_key'] = $security_key;
        
        $d['ipaddress'] = $ipaddress;
        unset($d['action']);
        unset($d['nonce']); // No enviar nonce a NMI
        
        // Agregar orden de WordPress
        if ( ! isset($d['orderid']) || empty($d['orderid']) ) {
            $d['orderid'] = $order_id;
        }

        if(isset($d['xid']) && $d['xid'] == 'null'){
            $d['xid'] = null;
        }

        foreach($d as $key => $row){
            if($row == null){
                $query.= $key.'='.($row).'&';
            }
            else{
                $query.= $key.'='.rawurlencode($row).'&';
            }
        }

        $url = "https://secure.nmi.com/api/transact.php";
        $args = [
            'sslverify' => false,
            'body' => $query,
        ];
        // $query = substr($query,0,-1);
        // $ch = curl_init();
        // curl_setopt($ch, CURLOPT_URL, "https://secure.nmi.com/api/transact.php");
        // curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 30);
        // curl_setopt($ch, CURLOPT_TIMEOUT, 30);
        // curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
        // curl_setopt($ch, CURLOPT_HEADER, 0);
        // curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, 0);

        // curl_setopt($ch, CURLOPT_POSTFIELDS, $query);
        // curl_setopt($ch, CURLOPT_POST, 1);

        // if (!($data = curl_exec($ch))) {
        //     return ERROR;
        // }
        // curl_close($ch);
        // unset($ch);

        $response = wp_remote_post('https://secure.nmi.com/api/transact.php', array(
            'method' => 'POST',
            'timeout' => 30,
            'body' => $query,
        ));
        // ===================== NUEVO: Manejo robusto de error HTTP =====================
        if ( is_wp_error($response) ) {
            $err = $response->get_error_message();
            // Intentar registrar en la orden para auditoria (si viene order_id)
            if ( ! empty($order_id) ) {
                $order_tmp = wc_get_order($order_id);
                if ( $order_tmp ) {
                    $order_tmp->add_order_note('ERROR BAC/NMI (wp_remote_post): ' . $err, 1);
                }
            }
            // Devolvemos un formato "querystring" compatible con el JS (res.response == 3)
            echo 'response=3&responsetext=' . urlencode('WP_ERROR: ' . $err);
            wp_die();
        }

        $http_code = wp_remote_retrieve_response_code($response);
        if ( $http_code < 200 || $http_code >= 300 ) {
            $body_tmp = wp_remote_retrieve_body($response);
            if ( ! empty($order_id) ) {
                $order_tmp = wc_get_order($order_id);
                if ( $order_tmp ) {
                    $order_tmp->add_order_note('ERROR BAC/NMI (HTTP ' . $http_code . '): ' . substr($body_tmp, 0, 300), 1);
                }
            }
            echo 'response=3&responsetext=' . urlencode('HTTP_ERROR: ' . $http_code);
            wp_die();
        }
        // ==============================================================================


        //$result = wp_remote_post($url, $args);

        $body = wp_remote_retrieve_body( $response );

        // ============ NUEVO: GUARDAR RESPUESTA EN WORDPRESS ============
        // Defaults para UI (aunque no exista la orden)
        $minimal_reason = 'Detalle no disponible';
        $ui_reason_key  = 'generalError';

        // Obtener la orden de WooCommerce
        $order = wc_get_order( $order_id );
        
        if ( $order ) {
            // Parsear la respuesta del BAC (formato: key=value&key=value)
            parse_str($body, $response_array);
            
            // Extraer datos importantes
            $transaction_id = isset($response_array['transactionid']) ? $response_array['transactionid'] : '';
            $response_code = isset($response_array['response']) ? $response_array['response'] : '';
            $response_text = isset($response_array['responsetext']) ? $response_array['responsetext'] : '';
            $auth_code = isset($response_array['authcode']) ? $response_array['authcode'] : '';
            
            // Nota interna minimalista (trazabilidad)
            $status = isset($response_array['response']) ? strval($response_array['response']) : strval($response_code);
            $minimal_reason = wolk_minimalize_payment_reason($response_text);
            $ui_reason_key  = wolk_map_ui_reason_key($response_text);

            // Evitar duplicados: si el mismo motivo ya fue agregado recientemente, no repetir.
// Incluir firma de tarjeta (marca/last4/cardholder) para no deduplicar entre tarjetas distintas.
list($card_audit_text, $card_audit_sig, $card_meta) = wolk_build_card_audit_string_from_post($_POST);

// Guardar metadatos (sin PAN completo)
if (!empty($card_meta['last4'])) $order->update_meta_data('_wolk_card_last4', $card_meta['last4']);
if (!empty($card_meta['brand'])) $order->update_meta_data('_wolk_card_brand', $card_meta['brand']);
if (!empty($card_meta['cardholder'])) $order->update_meta_data('_wolk_cardholder_name', $card_meta['cardholder']);

$note_key = md5($status . '|' . $minimal_reason . '|' . $card_audit_sig);
            $last_key = $order->get_meta('_wolk_last_note_key');
            $last_ts  = intval($order->get_meta('_wolk_last_note_ts'));
            $now_ts   = time();
            $is_duplicate = (!empty($last_key) && $last_key === $note_key && ($now_ts - $last_ts) < 300);

			// Trazabilidad: deviceChannel usado en el flujo (si viene desde el front)
			$device_channel = '';
			if (isset($_POST['deviceChannel'])) {
				$device_channel = sanitize_text_field($_POST['deviceChannel']);
			} elseif (isset($_POST['device_channel'])) {
				$device_channel = sanitize_text_field($_POST['device_channel']);
			}
			$device_text = $device_channel ? (' | deviceChannel=' . $device_channel) : '';

			if ($status === '1') {
                if (!$is_duplicate) {
                    $order->add_order_note(
						sprintf('Pago aprobado (BAC). ID Transaccion: %s%s%s', $transaction_id, $device_text, $card_audit_text),
                        1
                    );
                }
            } elseif ($status === '2') {
                if (!$is_duplicate) {
                    $order->add_order_note(
                        sprintf('Pago rechazado: %s%s', $minimal_reason, $card_audit_text),
                        1
                    );
                }
            } else {
                if (!$is_duplicate) {
                    $order->add_order_note(
                        sprintf('Error de pago: %s%s', $minimal_reason, $card_audit_text),
                        1
                    );
                }
            }

            if (!$is_duplicate) {
                $order->update_meta_data('_wolk_last_note_key', $note_key);
                $order->update_meta_data('_wolk_last_note_ts', $now_ts);
            }
            
            // OPCION 2: Guardar como meta de la orden (para acceso programatico)
            $order->update_meta_data( '_wolk_transaction_id', $transaction_id );
            $order->update_meta_data( '_wolk_response_code', $response_code );
            $order->update_meta_data( '_wolk_response_text', $response_text );
            $order->update_meta_data( '_wolk_auth_code', $auth_code );
            $order->update_meta_data( '_wolk_full_response', wp_json_encode($response_array) );
            $order->update_meta_data( '_wolk_response_date', current_time('mysql') );
            
            // Guardar los cambios
            $order->save();
            
            error_log('WOLK: Respuesta del BAC guardada en orden #' . $order_id);
            error_log('WOLK: Transaction ID: ' . $transaction_id);
            error_log('WOLK: Response Code: ' . $response_code);
        }
        
        // ============ FIN GUARDAR RESPUESTA ============

        // [OK] Enriquecer respuesta para UI: enviar clave y motivo ya clasificado (evita que el front muestre genericos)
        if (is_string($body)) {
            $append = '';
            $append .= '&wolk_reason_key=' . urlencode($ui_reason_key);
            $append .= '&wolk_reason=' . urlencode($minimal_reason);
            $body .= $append;
        }

        echo esc_html($body);
        exit();
}


// ============================================================
// FUNCION AUXILIAR: Detectar marca de tarjeta por PAN (solo para last4/marca; NO guardar PAN completo)
// ============================================================
function wolk_detect_card_brand($pan_digits) {
    $pan_digits = preg_replace('/\D+/', '', (string)$pan_digits);
    if ($pan_digits === '') return '';

    // AMEX: 34, 37 (15)
    if (preg_match('/^3[47]\d{13}$/', $pan_digits)) return 'AMEX';

    // VISA: 4 (13/16/19)
    if (preg_match('/^4\d{12}(\d{3})?(\d{3})?$/', $pan_digits)) return 'VISA';

    // MASTERCARD: 51-55 o 2221-2720 (16)
    if (preg_match('/^(5[1-5]\d{14}|2(2(2[1-9]|[3-9]\d)|[3-6]\d\d|7(0\d|1\d|20))\d{12})$/', $pan_digits)) return 'MASTERCARD';

    // DISCOVER: 6011, 65, 644-649
    if (preg_match('/^(6011\d{12}|65\d{14}|64[4-9]\d{13})$/', $pan_digits)) return 'DISCOVER';

    // DINERS: 300-305, 36, 38-39
    if (preg_match('/^(30[0-5]\d{11}|36\d{12}|3[89]\d{12})$/', $pan_digits)) return 'DINERS';

    // JCB: 3528-3589
    if (preg_match('/^35(2[89]|[3-8]\d)\d{12}$/', $pan_digits)) return 'JCB';

    return '';
}

// ============================================================
// FUNCION AUXILIAR: Construir string de auditoria de tarjeta (marca + last4 + cardholder)
// ============================================================
function wolk_build_card_audit_string_from_post($post) {
    $pan = '';
    if (isset($post['ccnumber'])) $pan = $post['ccnumber'];
    elseif (isset($post['cardNumber'])) $pan = $post['cardNumber'];

    $pan_digits = preg_replace('/\D+/', '', (string)$pan);
    $last4 = $pan_digits !== '' ? substr($pan_digits, -4) : '';

    // 1) Prioridad: titular real enviado desde el input "Titular de la tarjeta" (puede diferir del comprador)
    //    JS lo envia como cc_name.
    $cc_name = isset($post['cc_name']) ? sanitize_text_field($post['cc_name']) : '';
    $cc_name = trim($cc_name);

    // 2) Fallback: nombre/apellido del formulario (cliente/comprador)
    $first = isset($post['firstname']) ? sanitize_text_field($post['firstname']) : (isset($post['firstName']) ? sanitize_text_field($post['firstName']) : '');
    $last  = isset($post['lastname'])  ? sanitize_text_field($post['lastname'])  : (isset($post['lastName'])  ? sanitize_text_field($post['lastName'])  : '');
    $fallback_name = trim($first . ' ' . $last);

    $cardholder = $cc_name !== '' ? $cc_name : $fallback_name;

    $brand = wolk_detect_card_brand($pan_digits);

    // Guardar metadatos (opcionales) sin PAN completo
    $meta = [
        'brand' => $brand,
        'last4' => $last4,
        'cardholder' => $cardholder,
    ];

    // String para nota: (MASTERCARD | **** 1234 | Fabian Perez)
    $parts = [];
    if ($brand) $parts[] = $brand;
    if ($last4) $parts[] = '**** ' . $last4;
    if ($cardholder) $parts[] = $cardholder;

    $text = $parts ? (' (' . implode(' | ', $parts) . ')') : '';
    $sig  = implode('|', $parts); // para dedupe

    return [$text, $sig, $meta];
}

// ============================================================
// FUNCION AUXILIAR: Motivo minimalista para notas del pedido
// ============================================================
function wolk_minimalize_payment_reason($response_text) {
    $text = is_string($response_text) ? trim($response_text) : '';
    if ( $text === '' ) {
        return 'Detalle no disponible';
    }

    $t = strtolower($text);

    // Tarjeta bloqueada/suspendida (escenarios de pruebas o bloqueo del emisor)
    if (strpos($t, 'frozen') !== false || strpos($t, 'blocked') !== false || strpos($t, 'restricted') !== false || strpos($t, 'suspend') !== false) {
        return 'Tarjeta bloqueada';
    }

    // Heuristica (NMI/BAC responde mensajes en ingles con variantes)
    if (strpos($t, 'insufficient') !== false || strpos($t, 'fund') !== false) {
        return 'Fondos insuficientes';
    }
    // CVV/CVC (muchas variantes: CVV2, security code, mismatch, verification)
    if (
        strpos($t, 'cvv') !== false || strpos($t, 'cvc') !== false || strpos($t, 'cvv2') !== false || strpos($t, 'cvc2') !== false ||
        strpos($t, 'security code') !== false || strpos($t, 'verification') !== false || strpos($t, 'mismatch') !== false
    ) {
        return 'CVV invalido';
    }
    if (strpos($t, 'expired') !== false) {
        return 'Tarjeta vencida';
    }
    if (
        (strpos($t, 'invalid') !== false && (strpos($t, 'card') !== false || strpos($t, 'credit') !== false)) ||
        strpos($t, 'invalid card') !== false || strpos($t, 'card number') !== false || strpos($t, 'bad card') !== false
    ) {
        return 'Detalles de tarjeta no validos';
    }
    if (strpos($t, 'do not honor') !== false || strpos($t, 'declin') !== false) {
        return 'Transaccion rechazada';
    }

    // Fallback corto
    $clean = wp_strip_all_tags($text);
    if (strlen($clean) > 120) {
        $clean = substr($clean, 0, 120) . '...';
    }
    return $clean;
}

// ============================================================
// FUNCION AUXILIAR: Clave de motivo para UI (coincide con lang.php)
// ============================================================
function wolk_map_ui_reason_key($response_text) {
    $text = is_string($response_text) ? trim($response_text) : '';
    if ($text === '') {
        return 'generalError';
    }

    $t = strtolower($text);

    if (strpos($t, 'frozen') !== false || strpos($t, 'blocked') !== false || strpos($t, 'restricted') !== false || strpos($t, 'suspend') !== false ||
        strpos($t, 'bloquead') !== false || strpos($t, 'suspendid') !== false || strpos($t, 'restringid') !== false) {
        return 'cardFrozen';
    }
    if (strpos($t, 'insufficient') !== false || strpos($t, 'fund') !== false || strpos($t, 'fondos') !== false || strpos($t, 'saldo') !== false || strpos($t, 'insuficiente') !== false) {
        return 'insufficientFunds';
    }
    if (strpos($t, 'cvv') !== false || strpos($t, 'cvc') !== false || strpos($t, 'cvv2') !== false || strpos($t, 'cvc2') !== false ||
        strpos($t, 'security code') !== false || strpos($t, 'verification') !== false || strpos($t, 'mismatch') !== false ||
        strpos($t, 'codigo de seguridad') !== false || strpos($t, 'codigo de seguridad') !== false) {
        return 'cvvInvalid';
    }
    if (strpos($t, 'expired') !== false) {
        return 'cardExpired';
    }
    return 'generalError';
}

// ============================================================
// FUNCION AUXILIAR: Obtener datos de la respuesta del BAC
// ============================================================

/**
 * Obtener datos guardados de la transaccion del BAC
 * Uso: $data = wolk_get_transaction_data( $order_id );
 */
function wolk_get_transaction_data( $order_id ) {
    $order = wc_get_order( $order_id );
    
    if ( ! $order ) {
        return false;
    }
    
    return array(
        'transaction_id' => $order->get_meta( '_wolk_transaction_id' ),
        'response_code' => $order->get_meta( '_wolk_response_code' ),
        'response_text' => $order->get_meta( '_wolk_response_text' ),
        'auth_code' => $order->get_meta( '_wolk_auth_code' ),
        'full_response' => $order->get_meta( '_wolk_full_response' ),
        'response_date' => $order->get_meta( '_wolk_response_date' ),
    );
}

// ============================================================
// AJAX: Agregar nota interna minimalista al pedido
// ============================================================
function wolk_add_order_note() {
    if ( !isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'ajax-nonce') ) {
        wp_die('Invalid Nonce!');
    }

    $order_id = isset($_POST['order_id']) ? absint($_POST['order_id']) : 0;
    $message  = isset($_POST['message']) ? sanitize_text_field($_POST['message']) : '';

    if ( !$order_id || empty($message) ) {
        wp_die();
    }

    $order = wc_get_order($order_id);
    if ( $order ) {
        // Nota privada (no visible al cliente)
        $order->add_order_note($message, 1);
        $order->save();
    }

    wp_die();
}

add_action('wp_ajax_wolk_add_order_note', 'wolk_add_order_note');
add_action('wp_ajax_nopriv_wolk_add_order_note', 'wolk_add_order_note');

add_action('wp_ajax_wolk_submit_payment_request', 'wolk_submit_payment_request' ); // executed when logged in
add_action('wp_ajax_nopriv_wolk_submit_payment_request', 'wolk_submit_payment_request' ); // executed when logged out



add_action( 'wp_ajax_getpostsfordatatables', 'wolk_ajax_getpostsfordatatables' );
add_action( 'wp_ajax_nopriv_getpostsfordatatables', 'wolk_ajax_getpostsfordatatables' );
function wolk_ajax_getpostsfordatatables() {
    global $wpdb;
    $wolk_error_logs_table_name = sprintf('%s' . 'wolk_error_logs', $wpdb->prefix);

    $start = isset($_GET['start']) ? absint($_GET['start']) : 1;
    $escaped_table_name = esc_sql($wolk_error_logs_table_name);
    $total = $wpdb->get_var($wpdb->prepare("SELECT COUNT(*) FROM {$escaped_table_name}"));
    $limit = isset($_GET['length']) ? absint($_GET['length']) : 10;

    $search = esc_attr(trim(sanitize_text_field($_GET['search']['value'] ?? '')));
    $search_sql = "";

    if ($search !== "") {
        $search_term = '%' . $wpdb->esc_like($search) . '%';
        $search_sql .= $wpdb->prepare(
            " WHERE order_id LIKE %s OR cc_last_6 LIKE %s OR cc_name LIKE %s OR type LIKE %s ",
            $search_term, $search_term, $search_term, $search_term
        );
    }

    $result = $wpdb->get_results(
        $wpdb->prepare(
            "SELECT * FROM {$escaped_table_name} {$search_sql} ORDER BY created_time DESC LIMIT %d, %d",
            $start, $limit
        )
    );

    $data = [];

    foreach ($result as $row) {
        $sub_data = [];

        // =========================
        // PROCESAMIENTO DETAILS
        // =========================
        $details_data = '';

        // Intentar decodificar JSON
        $details = json_decode($row->details, true);

        if (json_last_error() === JSON_ERROR_NONE && !empty($details)) {
            // JSON valido -> convertir en texto plano
            $details_data = esc_html(print_r($details, true));
        } else {
            // No es JSON valido o esta vacio
            $raw = trim($row->details);

            if ($raw === '' || $raw === '{}' || $raw === '[]') {
                $details_data = 'No hay detalles disponibles';
            } else {
                // Escapar siempre -> evitar HTML inyectado
                $details_data = esc_html($raw);

                // Limitar largo excesivo (evita congelar tabla)
                if (strlen($details_data) > 600) {
                    $details_data = substr($details_data, 0, 600) . '...';
                }
            }
        }

        // =========================
        // SANITIZACION DE CAMPOS
        // =========================
        $order_id  = !empty($row->order_id)  ? esc_html($row->order_id)  : 'N/A';
        $cc_last_6 = !empty($row->cc_last_6) ? esc_html($row->cc_last_6) : 'N/A';
        $cc_name   = !empty($row->cc_name)   ? esc_html($row->cc_name)   : 'N/A';
        $type      = !empty($row->type)      ? esc_html($row->type)      : 'unknown';
        $created   = esc_html($row->created_time);

        // =========================
        // ARMADO DE LA FILA
        // =========================
        $sub_data[] = $order_id;
        $sub_data[] = $cc_last_6;
        $sub_data[] = $cc_name;
        $sub_data[] = $details_data;  // puro texto seguro
        $sub_data[] = $type;
        $sub_data[] = $created;

        $data[] = $sub_data;
    }

    // =========================
    // RESPUESTA FINAL
    // =========================
    echo wp_json_encode([
        "draw"            => isset($_GET["draw"]) ? absint($_GET["draw"]) : 0,
        "data"            => $data,
        "recordsTotal"    => intval($total),
        "recordsFiltered" => intval(count($result)),
    ]);

    wp_die();
}



add_action('wp_ajax_wolk_submit_error_logs', 'wolk_submit_error_logs' );
add_action('wp_ajax_nopriv_wolk_submit_error_logs', 'wolk_submit_error_logs' );
function wolk_submit_error_logs() {
    global $wpdb;
    $wolk_error_logs_table_name = $wpdb->prefix . 'wolk_error_logs';

    // SEGURIDAD: Validar nonce
    if ( ! isset($_POST['nonce']) || ! wp_verify_nonce( wp_unslash($_POST['nonce']), 'ajax-nonce' ) ) {
        error_log('[WOLK] Error: Invalid Nonce en wolk_submit_error_logs');
        wp_send_json_error(['message' => 'Invalid Nonce!']);
        wp_die();
    }
    
    // SEGURIDAD PCI: Rate limiting por IP para prevenir flooding
    $client_ip = isset($_SERVER['REMOTE_ADDR']) ? sanitize_text_field($_SERVER['REMOTE_ADDR']) : 'unknown';
    $rate_key = 'wolk_log_rate_' . md5($client_ip);
    $attempts = (int) get_transient($rate_key);
    
    if ($attempts > 20) { // Maximo 20 logs por minuto por IP
        wp_send_json_error(['message' => 'Demasiadas solicitudes. Intente mas tarde.']);
        wp_die();
    }
    
    set_transient($rate_key, $attempts + 1, MINUTE_IN_SECONDS);

    // Extraer y validar datos con valores por defecto
    $order_id = isset($_POST['order_id']) && !empty($_POST['order_id']) 
        ? sanitize_text_field(wp_unslash($_POST['order_id'])) 
        : 'NO_ORDER_ID';
    
    // SEGURIDAD PCI: Sanitizar cc_last_6 - solo digitos, maximo 6
    $cc_last_6_raw = isset($_POST['cc_last_6']) ? wp_unslash($_POST['cc_last_6']) : '';
    $cc_last_6 = preg_replace('/[^0-9]/', '', $cc_last_6_raw);
    $cc_last_6 = substr($cc_last_6, 0, 6);
    if (empty($cc_last_6)) {
        $cc_last_6 = 'NOT_PROVIDED';
    }
    
    $cc_name = isset($_POST['cc_name']) && !empty($_POST['cc_name']) 
        ? sanitize_text_field(wp_unslash($_POST['cc_name'])) 
        : 'NOT_PROVIDED';
    
    // SEGURIDAD PCI: Sanitizar y validar details como JSON
    $details_raw = isset($_POST['details']) ? wp_unslash($_POST['details']) : '{}';
    $details_decoded = json_decode($details_raw, true);
    
    if (json_last_error() === JSON_ERROR_NONE && is_array($details_decoded)) {
        // SEGURIDAD PCI: Remover cualquier dato sensible que pueda haberse filtrado
        unset($details_decoded['ccnumber']);
        unset($details_decoded['cvv']);
        unset($details_decoded['security_key']);
        unset($details_decoded['ccexp']);
        unset($details_decoded['cvc']);
        $details = wp_json_encode($details_decoded);
    } else {
        // No es JSON valido, sanitizar como texto plano
        $details = sanitize_textarea_field($details_raw);
    }
    
    // SEGURIDAD: Limitar longitud
    if (strlen($details) > 5000) {
        $details = substr($details, 0, 5000) . '...[truncated]';
    }
    
    $type = isset($_POST['type']) && !empty($_POST['type']) 
        ? substr(sanitize_text_field(wp_unslash($_POST['type'])), 0, 50)
        : 'unknown';

    // Log para debug (solo en entorno de desarrollo)
    if (defined('WP_DEBUG') && WP_DEBUG) {
        error_log('[WOLK] Guardando error log: ' . wp_json_encode([
            'order_id' => $order_id,
            'cc_last_6' => $cc_last_6,
            'cc_name' => $cc_name,
            'type' => $type,
            'details_length' => strlen($details)
        ]));
    }

    // Preparar datos para insertar
    $data = array(
        'order_id' => $order_id,
        'cc_last_6' => $cc_last_6,
        'cc_name' => $cc_name,
        'details' => $details,
        'type' => $type,
    );
    
    // SEGURIDAD: Usar formatos especificos
    $formats = array('%s', '%s', '%s', '%s', '%s');

    // Insertar en la base de datos
    $result = $wpdb->insert($wolk_error_logs_table_name, $data, $formats);

    // Verificar resultado
    if ($result === false) {
        error_log('[WOLK] Error al insertar en BD: ' . $wpdb->last_error);
        error_log('[WOLK] Datos que se intentaron insertar: ' . print_r($data, true));
        wp_send_json_error([
            'message' => 'Database error', 
            'sql_error' => $wpdb->last_error
        ]);
    } else {
        if (defined('WP_DEBUG') && WP_DEBUG) {
            error_log('[WOLK] Log guardado exitosamente con ID: ' . $wpdb->insert_id);
        }
        wp_send_json_success([
            'message' => 'Log guardado exitosamente', 
            'insert_id' => $wpdb->insert_id
        ]);
    }

    wp_die();
}

// Maneja la solicitud AJAX para desencriptar el processor_id
add_action('wp_ajax_decrypt_processor_id', 'decrypt_processor_id_callback');
add_action('wp_ajax_nopriv_decrypt_processor_id', 'decrypt_processor_id_callback');

function decrypt_processor_id_callback() {
    // SEGURIDAD: Verificar nonce PRIMERO
    if (!isset($_POST['nonce']) || !wp_verify_nonce(wp_unslash($_POST['nonce']), 'ajax-nonce')) {
        wp_send_json_error(['message' => 'Solicitud no valida.']);
        return;
    }
    
    // Validar que el order_id existe
    if (!isset($_POST['order_id']) || empty($_POST['order_id'])) {
        wp_send_json_error(['message' => 'ID de la orden no proporcionado.']);
        return;
    }

    // SEGURIDAD: usar absint() para order_id
    $order_id = absint($_POST['order_id']);
    $order = wc_get_order($order_id);

    if (!$order) {
        wp_send_json_error(['message' => 'Orden no encontrada o invalida.']);
        return;
    }
    
    // SEGURIDAD: Verificar que la orden esta en estado valido para pago
    if (!$order->has_status(['pending', 'on-hold', 'failed'])) {
        wp_send_json_error(['message' => 'Esta orden no puede ser procesada.']);
        return;
    }
    
    // SEGURIDAD: Verificar propiedad de la orden (para usuarios logueados)
    $current_user_id = get_current_user_id();
    $order_customer_id = $order->get_customer_id();
    
    if ($current_user_id > 0 && $order_customer_id > 0 && $current_user_id !== $order_customer_id) {
        wp_send_json_error(['message' => 'No autorizado.']);
        return;
    }

    // Obtener y validar el processor_id encriptado
    $encrypted_processor_id = isset($_POST['encrypted_processor_id']) 
        ? sanitize_text_field(wp_unslash($_POST['encrypted_processor_id'])) 
        : '';
        
    if (empty($encrypted_processor_id)) {
        wp_send_json_error(['message' => 'Processor ID no proporcionado.']);
        return;
    }

    // Manejar el caso de pago unico
    if ($encrypted_processor_id === 'single_payment') {
        wp_send_json_success(['processor_id' => null]);
        return;
    }

    // Crear instancia de Wolk_Payment_Gateway
    $Payment_Gateway = new Wolk_Payment_Gateway();

    // Desencriptar el processor_id
    $processor_id = $Payment_Gateway->encrypt_decrypt('decrypt', $encrypted_processor_id, 'order', $order_id);

    if (!$processor_id) {
        wp_send_json_error(['message' => 'Error al procesar el plan de pago.']);
        return;
    }

    // Responder con el processor_id desencriptado
    wp_send_json_success(['processor_id' => $processor_id]);
}

// ============================================================================
// SISTEMA DE AUTO-ACTUALIZACIÓN - VERSIÓN SIMPLIFICADA
// ============================================================================

/**
 * Verificar actualizaciones disponibles
 */
function wolk_payment_check_for_updates() {
    $remote_url = WOLK_PAYMENT_UPDATE_CHECK_URL;
    
    $response = wp_remote_get($remote_url, array(
        'timeout' => 10,
        'sslverify' => true
    ));
    
    if (is_wp_error($response)) {
        return false;
    }
    
    $body = wp_remote_retrieve_body($response);
    $remote_data = json_decode($body, true);
    
    if (!$remote_data || !isset($remote_data['date']) || !isset($remote_data['version'])) {
        return false;
    }
    
    $current_date = strtotime(WOLK_PAYMENT_LAST_UPDATE);
    $remote_date = strtotime($remote_data['date']);
    
    if ($remote_date > $current_date) {
        return array(
            'new_version' => $remote_data['version'],
            'current_version' => WOLK_PAYMENT_VERSION,
            'date' => $remote_data['date'],
            'download_url' => isset($remote_data['download_url']) ? $remote_data['download_url'] : '',
            'changelog' => isset($remote_data['changelog']) ? $remote_data['changelog'] : ''
        );
    }
    
    return false;
}

/**
 * Mostrar aviso de actualización - VERSIÓN SIMPLE
 */
function wolk_payment_update_notice() {
    if (!current_user_can('manage_options')) {
        return;
    }
    
    $update_data = get_transient('wolk_payment_update_check');
    
    if (false === $update_data) {
        $update_data = wolk_payment_check_for_updates();
        set_transient('wolk_payment_update_check', $update_data ? $update_data : 'none', 12 * HOUR_IN_SECONDS);
    }
    
    if (!$update_data || $update_data === 'none' || !is_array($update_data)) {
        return;
    }
    
    echo '<div class="notice notice-warning is-dismissible">';
    echo '<p><strong>🔔 Wolk Payment Cardinal - Nueva Versión Disponible</strong></p>';
    echo '<p>Versión actual: ' . esc_html($update_data['current_version']) . ' (' . esc_html(WOLK_PAYMENT_LAST_UPDATE) . ')</p>';
    echo '<p>Nueva versión: <strong>' . esc_html($update_data['new_version']) . '</strong> (' . esc_html($update_data['date']) . ')</p>';
    
    if (!empty($update_data['changelog'])) {
        echo '<p>Cambios: ' . esc_html($update_data['changelog']) . '</p>';
    }
    
    if (!empty($update_data['download_url'])) {
        echo '<p><a href="' . esc_url($update_data['download_url']) . '" class="button button-primary">📥 Descargar</a></p>';
    }
    
    echo '</div>';
}

add_action('admin_notices', 'wolk_payment_update_notice');

function wolk_payment_force_update_check() {
    delete_transient('wolk_payment_update_check');
}

register_deactivation_hook(__FILE__, 'wolk_payment_force_update_check');

// ============================================================================
// FIN SISTEMA DE AUTO-ACTUALIZACIÓN
// ============================================================================


/* datatables css and js */



